<?php

class Page
{
    public const SCRIPT_ASSET = '1';
    public const STYLE_ASSET = '2';

    private $assets = [];
    private $breadCrumbs = [];
    private $title;
    private $description;
    private $keywords;
    private $image;

    public function __construct()
    {
        $this->title = page()->title;
        $this->description = page()->description;
        $this->keywords = page()->keywords;
        $this->image = page()->image;

        return $this;
    }

    public function isAdminPage()
    {
        return 1 != page()->type;
    }

    public function setAsset($assetUrl, $assetType)
    {
        $this->assets[] = [
            'url' => $assetUrl,
            'type' => $assetType,
        ];

        return $this;
    }

    public function collectPage($template, $data = [], $shieldingIgnore = [])
    {
        $this->collectTitle();
        $this->collectHead();

        if ($this->isAdminPage()) {
            if (!is_admin()) {
                show_error_page('not_adm');
            }

            $this->collectAdminPage($template, $data, $shieldingIgnore);
        } else {
            if ((1 == page()->privacy || 0 == page()->privacy) && !is_auth()) {
                show_error_page('not_auth');
            }

            $this->collectUserPage($template, $data, $shieldingIgnore);
        }

        return $this;
    }

    public function setBreadCrumbs($breadCrumbs)
    {
        $this->breadCrumbs = $breadCrumbs;

        return $this;
    }

    public function setTitle($title)
    {
        $this->title = $title;

        return $this;
    }

    public function substituteToTitle($value)
    {
        $this->title = str_replace('{value}', $value, page()->title);

        return $this;
    }

    public function setDescription($description)
    {
        $this->description = $description;

        return $this;
    }

    public function substituteToDescription($description)
    {
        $this->description = str_replace('{value}', $description, page()->description);

        return $this;
    }

    public function setKeywords($keywords)
    {
        $this->keywords = $keywords;

        return $this;
    }

    public function substituteToKeywords($keywords)
    {
        $this->keywords = str_replace('{value}', $keywords, page()->keywords);

        return $this;
    }

    public function setKeywordsFromString($string)
    {
        $this->keywords = str_replace(
            ',,',
            ',',
            str_replace(' ', ',', $string)
        );

        return $this;
    }

    public function setImage($image)
    {
        $this->image = $image;

        return $this;
    }

    private function getAssetHtml()
    {
        $assetsHtml = '';

        foreach ($this->assets as $adminAsset) {
            if (self::SCRIPT_ASSET == $adminAsset['type']) {
                $assetsHtml .= '<script src="'.$adminAsset['url'].'?v={cache}"></script>';
            }

            if (self::STYLE_ASSET == $adminAsset['type']) {
                $assetsHtml .= '<link rel="stylesheet" href="'.$adminAsset['url'].'?v={cache}">';
            }
        }

        return $assetsHtml;
    }

    private function collectUserPage($templateFile, $data, $shieldingIgnore)
    {
        $nav = tpl()->get_nav(
            $this->breadCrumbs,
            'elements/nav_li.tpl',
            1
        );

        if (is_auth()) {
            include_once __DIR__.'/../authorized.php';
        } else {
            include_once __DIR__.'/../not_authorized.php';
        }

        $this->collectBody($templateFile, $data, $shieldingIgnore);
    }

    private function collectAdminPage($templateFile, $data, $shieldingIgnore)
    {
        tpl()->load_template('top.tpl');
        tpl()->set('{site_name}', configs()->name);
        tpl()->compile('content');
        tpl()->clear();

        tpl()->load_template('menu.tpl');
        tpl()->set('{site_host}', '../');
        tpl()->compile('content');
        tpl()->clear();

        $breadCrumbsNav = tpl()->get_nav(
            $this->breadCrumbs,
            'elements/nav_li.tpl',
            1
        );

        tpl()->load_template('page_top.tpl');
        tpl()->set('{nav}', $breadCrumbsNav);
        tpl()->compile('content');
        tpl()->clear();

        $this->collectBody($templateFile, $data, $shieldingIgnore);

        tpl()->load_template('bottom.tpl');
        tpl()->set('{site_host}', '../');
        tpl()->compile('content');
        tpl()->clear();
    }

    private function collectTitle()
    {
        tpl()->load_template('elements/title.tpl');
        tpl()->set('{title}', $this->title);
        tpl()->set('{name}', configs()->name);
        tpl()->compile('title');
        tpl()->clear();
    }

    private function collectHead()
    {
        tpl()->load_template('head.tpl');
        tpl()->set('{title}', tpl()->result['title']);
        tpl()->set('{site_name}', configs()->name);
        tpl()->set('{image}', $this->image);
        tpl()->set('{robots}', page()->robots);
        tpl()->set('{type}', page()->kind);
        tpl()->set('{description}', $this->description);
        tpl()->set('{keywords}', $this->keywords);
        tpl()->set('{url}', page()->full_url);
        tpl()->set('{other}', $this->getAssetHtml());
        tpl()->set('{token}', token());
        tpl()->set('{template}', get_template());
        tpl()->set('{site_host}', '../');
        tpl()->compile('content');
        tpl()->clear();
    }

    private function collectBody($templateFile, $data = [], $shieldingIgnore = [])
    {
        if (page()->module) {
            $module = (new ExtraModule())->get(page()->module);

            if ($this->isAdminPage()) {
                $templateDir = Template::DOWN_TO_ROOT.tpl()->getRelativeExtraModuleAdminDir($module->name);
            } else {
                $templateDir = Template::DOWN_TO_ROOT.tpl()->getRelativeExtraModuleDir($module->name);
            }

            $templateFile = $templateDir.$templateFile;
        }

        tpl()->load_template($templateFile);

        $data['site_host'] = '../';
        foreach ($data as $key => $value) {
            if (in_array($key, $shieldingIgnore)) {
                tpl()->set('{'.$key.'}', $value, false);
            } else {
                tpl()->set('{'.$key.'}', $value);
            }
        }

        tpl()->compile('content');
        tpl()->clear();
    }
}
