<?php

namespace GameCMS\Statistics;

use GameCMS\Models\Server;

class LevelsRanksIntegration extends DatabasedIntegration implements StatisticIntegration
{
    private \GetData $getData;

    private string $whereStatement = '(kills!=0)';

    private array $sortTypes = [
        Integrations::SORT_BY_RANK,
        Integrations::SORT_BY_POINTS,
    ];

    private array $defaultRanks = [
        0 => [
            'names' => [
                0 => 'Калибровка',
                1 => 'Серебро I',
                2 => 'Серебро II',
                3 => 'Серебро III',
                4 => 'Серебро IV',
                5 => 'Серебро-Элита',
                6 => 'Серебро-Великий Магистр',
                7 => 'Золотая Звезда I',
                8 => 'Золотая Звезда II',
                9 => 'Золотая Звезда III',
                10 => 'Золотая Звезда — Магистр',
                11 => 'Магистр-хранитель I',
                12 => 'Магистр-хранитель II',
                13 => 'Магистр-хранитель Элита',
                14 => 'Заслуженный Магистр-хранитель',
                15 => 'Легендарный Беркут',
                16 => 'Легендарный Беркут — Магистр',
                17 => 'Великий Магистр Высшего Ранга',
                18 => 'Всемирная Элита',
            ],
            'imagesFolder' => 'ranks_imgs',
        ],
    ];

    public function __construct()
    {
        $this->getData = new \GetData();
    }

    public function isBlank(): bool
    {
        return false;
    }

    public function isValidStatisticsSortType(int $type): bool
    {
        return in_array($type, $this->sortTypes);
    }

    public function isValidDataBase(\PDO $connection, string $table = ''): bool
    {
        if (empty($table)) {
            return false;
        }

        if (!check_table($table, $connection)) {
            return false;
        }

        if (
            !check_column($table, $connection, 'value')
            || !check_column($table, $connection, 'rank')
            || !check_column($table, $connection, 'steam')
        ) {
            return false;
        }

        return true;
    }

    public function removeStats(Server $server, \PDO $connection, string $playerId): bool
    {
        $steamId = preg_replace('/[^0-9]+/', '', $playerId);
        $steamId = 'STEAM_'.$steamId[1].':'.$steamId[2].':'.substr($steamId, 3);

        $STH = $connection->prepare("DELETE FROM {$server->st_db_table} WHERE steam=:id LIMIT 1");
        $STH->execute([':id' => $steamId]);

        return true;
    }

    public function getStatsCount(Server $server, \PDO $connection): int
    {
        return $connection
            ->query("SELECT COUNT(*) as count FROM {$server->st_db_table} WHERE {$this->whereStatement}")
            ->fetchColumn();
    }

    public function isHasRank(Server $server, \PDO $connection): bool
    {
        return false;
    }

    public function renderCommonStatistics(
        Server $server,
        \PDO $connection,
        \Template $tpl,
        string $name,
        int $start,
        int $limit
    ): string {
        $tpl->result['local_content'] = '';

        $isHasRank = $this->isHasRank($server, $connection);

        $ranks = getConfigAdditional('customRanks', $this->defaultRanks);
        $ranks[0] = $this->defaultRanks[0];

        $sort = $this->getSort($server->st_sort_type);

        if (check_column($server->st_db_table, $connection, 'weaponhits')) {
            $shoots = 'weaponshoots';
        } else {
            $shoots = 'shoots';
        }

        if (check_column($server->st_db_table, $connection, 'playtime')) {
            $playtime = 'playtime AS gametime, ';
        } else {
            $playtime = '';
        }

        if (empty($name)) {
            $STH = $connection->query(
                "SELECT 
					steam AS authid,
				    name AS nick,
				    deaths,
				    kills AS frags, 
				    hits,
				    lastconnect AS lasttime,
				    headshots,
				    value,
				    `rank`,
				    {$playtime} 
				    {$shoots} AS shots 
				FROM {$server->st_db_table}
				WHERE {$this->whereStatement}
				ORDER BY {$sort}
				LIMIT {$start}, {$limit}"
            );
        } else {
            $STH = $connection->prepare(
                prepareSearchQuery(
                    "SELECT 
						steam AS authid,
						name AS nick,
						deaths,
						kills AS frags,
						hits,
						lastconnect AS lasttime,
						headshots,
						value,
						`rank`, 
						{$playtime} 
						{$shoots} AS shots
					FROM {$server->st_db_table}
					WHERE {$this->whereStatement} 
					  	AND (name LIKE :name OR steam LIKE :name)
					ORDER BY {$sort}"
                )
            );
            $STH->execute([':name' => getNameLike($name)]);
        }

        while ($row = $STH->fetchObject()) {
            ++$start;

            $row->id = '9'.preg_replace('/[^0-9]+/', '', $row->authid);

            if (empty($row->authid)) {
                $row->authid = '';
            }

            if (empty($name)) {
                $row->place = $start;
            } else {
                $where = $this->getWhere($server->st_sort_type, $row->rank, $row->value);

                $STH2 = $connection->query(
                    "SELECT 
						count(*)
					FROM {$server->st_db_table}
					WHERE {$this->whereStatement} 
					  	AND ({$where})
					ORDER BY {$sort}"
                );
                $row->place = $STH2->fetchColumn() + 1;
            }

            if ($user_profile = $this->getData->get_gamer_profile($row->nick, $row->authid)) {
                $nick = $user_profile;
            } else {
                $nick = clean($row->nick, null);
            }

            $ranksId = 0;
            if (array_key_exists($server->id, $ranks)) {
                $ranksId = $server->id;
            }

            $rank_img = '../files/'.$ranks[$ranksId]['imagesFolder'].'/'.$row->rank.'.png?v={cache}';
            $rank_name = $ranks[$ranksId]['names'][$row->rank] ?? '';

            $tpl->load_template('elements/stat.tpl');
            $tpl->set('{authid}', isNeedHidePlayerId() ? hidePlayerId($row->authid) : $row->authid);
            $tpl->set('{authid_original}', $row->authid);
            $tpl->set('{time}', $row->gametime ? expand_seconds2($row->gametime) : '');
            $tpl->set('{value}', $row->value);
            $tpl->set('{rank_img}', $rank_img);
            $tpl->set('{rank_name}', $rank_name);
            $tpl->set('{type}', $server->st_type);
            $tpl->set('{id}', $row->id);
            $tpl->set('{place}', $row->place);
            $tpl->set('{nick}', $nick);
            $tpl->set('{frags}', $this->lvlRank($row->frags));
            $tpl->set('{deaths}', $this->lvlRank($row->deaths));
            $tpl->set('{headshots}', $this->lvlRank($row->headshots));
            $tpl->set('{shots}', $this->lvlRank($row->shots));
            $tpl->set('{hits}', $this->lvlRank($row->hits));
            $tpl->set('{kdr}', (0 == $row->deaths) ? $row->frags : round($row->frags / $row->deaths, 2));
            $tpl->set('{procent2}', get_procent($row->frags / 100, $row->headshots));
            $tpl->set('{procent4}', get_procent($row->shots / 100, $row->hits));
            $tpl->set('{date}', expand_date(date('d.m.Y H:i', $row->lasttime), 7));
            $tpl->set('{server}', $server->id);
            $tpl->set('{isHasRank}', $isHasRank);
            $tpl->compile('local_content');
            $tpl->clear();
        }

        if (empty($tpl->result['local_content'])) {
            return tableRowError('Нет результатов');
        }

        return $tpl->result['local_content'];
    }

    public function renderWeaponsStatistics(Server $server, \PDO $connection, \Template $tpl, string $playerId): string
    {
        return '';
    }

    public function renderMapsStatistics(Server $server, \PDO $connection, \Template $tpl, string $playerId): string
    {
        return '';
    }

    private function getSort($type): string
    {
        $sort = '';

        if (Integrations::SORT_BY_RANK == $type) {
            $sort = '`rank` DESC';
        } elseif (Integrations::SORT_BY_POINTS == $type) {
            $sort = 'value DESC';
        }

        return $sort;
    }

    private function getWhere(
        $type,
        $rank = null,
        $value = null
    ): string {
        $where = '';

        if (Integrations::SORT_BY_RANK == $type) {
            $where = '`rank` > '.$rank;
        } elseif (Integrations::SORT_BY_POINTS == $type) {
            $where = 'value > '.$value;
        }

        return $where;
    }

    private function lvlRank($value): string
    {
        if (false === strripos($value ?? '', ';')) {
            return $value;
        }
        $value = explode(';', $value);

        return trim($value[0]);
    }
}
