<?php

namespace GameCMS\Statistics;

use GameCMS\Models\Server;

class HlStatsxCeIntegration extends DatabasedIntegration implements StatisticIntegration
{
    private \GetData $getData;

    private string $whereStatement = '(kills!=0)';

    private array $defaultSkills = [0, 0.20, 0.40, 0.60, 0.80, 1.00, 1.20, 1.40, 1.60, 1.80, 2.00, 2.50];

    private array $defaultSkillNames = [
        'ULOW',
        'LOW-',
        'LOW',
        'LOW+',
        'MID-',
        'MID',
        'MID+',
        'HIGH-',
        'HIGH',
        'HIGH+',
        'PRO',
        'GOD',
    ];

    private array $skillsColors = [
        'default',
        'default',
        'default',
        'default',
        'success',
        'success',
        'success',
        'primary',
        'primary',
        'primary',
        'warning',
        'danger',
    ];

    private array $sortTypes = [
        Integrations::SORT_BY_KILLS_DEATHS_TEAMKILLS_DIFF,
        Integrations::SORT_BY_KILLS,
        Integrations::SORT_BY_KILLS_HEADSHODS_SUM,
        Integrations::SORT_BY_SKILL,
        Integrations::SORT_BY_ONLINE_TIME,
    ];

    public function __construct()
    {
        $this->getData = new \GetData();
    }

    public function isValidStatisticsSortType(int $type): bool
    {
        return in_array($type, $this->sortTypes);
    }

    public function isValidDataBase(\PDO $connection, string $table = ''): bool
    {
        if (
            !(check_table('hlstats_Players', $connection) && check_table('hlstats_Servers', $connection))
            && !(check_table('hlstats_players', $connection) && check_table('hlstats_servers', $connection))
        ) {
            return false;
        }

        return true;
    }

    public function removeStats(Server $server, \PDO $connection, string $playerId): bool
    {
        $STH = $connection->prepare('DELETE FROM hlstats_Players WHERE playerId=:id LIMIT 1');
        $STH->execute([':id' => $playerId]);

        return true;
    }

    public function getStatsCount(Server $server, \PDO $connection): int
    {
        $STH = $connection->prepare('SELECT game FROM hlstats_Servers WHERE address=:address AND port=:port LIMIT 1');
        $STH->execute([':address' => $server->ip, ':port' => $server->port]);
        $game = $STH->fetchColumn();
        if (empty($game)) {
            $game = 'csgo';
        }

        return $connection
            ->query("SELECT COUNT(*) as count FROM hlstats_Players WHERE {$this->whereStatement} AND game='{$game}'")
            ->fetchColumn();
    }

    public function isHasRank(Server $server, \PDO $connection): bool
    {
        return false;
    }

    public function isBlank(): bool
    {
        return false;
    }

    public function renderCommonStatistics(
        Server $server,
        \PDO $connection,
        \Template $tpl,
        string $name,
        int $start,
        int $limit
    ): string {
        $tpl->result['local_content'] = '';

        $isHasRank = $this->isHasRank($server, $connection);

        $STH = $connection->prepare(
            'SELECT game
			FROM hlstats_Servers
			WHERE address=:address
				AND port=:port
			LIMIT 1'
        );
        $STH->execute([':address' => $server->ip, ':port' => $server->port]);
        $game = $STH->fetchColumn();
        if (empty($game)) {
            $game = 'csgo';
        }

        $sort = $this->getSort($server->st_sort_type);

        if (empty($name)) {
            $STH = $connection->query(
                "SELECT 
					playerId AS id,
					kill_streak,
					death_streak,
					connection_time AS gametime,
					last_event AS lasttime,
					unhex(replace(hex(lastName), 'E280AE', '')) AS nick,
					skill,
					kills AS frags,
					deaths,
					suicides AS suicide,
					teamkills,
					shots,
					hits,
					headshots
				FROM hlstats_Players
				WHERE {$this->whereStatement}
				  	AND game='{$game}'
				ORDER BY {$sort}
				LIMIT {$start}, {$limit}"
            );
        } else {
            $STH = $connection->prepare(
                prepareSearchQuery(
                    "SELECT 
						playerId AS id,
					    kill_streak,
					    death_streak,
					    connection_time AS gametime,
					    last_event AS lasttime,
					    unhex(replace(hex(lastName), 'E280AE', '')) AS nick,
					    skill,
					    kills AS frags,
					    deaths,
					    suicides AS suicide,
					    teamkills,
					    shots,
					    hits,
					    headshots
					FROM hlstats_Players
					WHERE 
						{$this->whereStatement}
					  	AND game='{$game}' 
					  	AND (unhex(replace(hex(lastName), 'E280AE', '')) LIKE :name OR lastAddress LIKE :name)
					ORDER BY {$sort}"
                )
            );
            $STH->execute([':name' => getNameLike($name)]);
        }

        while ($row = $STH->fetchObject()) {
            ++$start;

            if (empty($row->authid)) {
                $row->authid = '';
            }

            if (empty($name)) {
                $row->place = $start;
            } else {
                $where = $this->getWhere(
                    $server->st_sort_type,
                    $row->frags,
                    $row->deaths,
                    $row->teamkills,
                    $row->headshots,
                    $row->skill,
                    $row->gametime,
                );

                $STH2 = $connection->query(
                    "SELECT 
						count(*)
					FROM hlstats_Players
					WHERE {$this->whereStatement}
					  	AND game='{$game}' AND ({$where})
					ORDER BY {$sort}"
                );
                $row->place = $STH2->fetchColumn() + 1;
            }

            if ($user_profile = $this->getData->get_gamer_profile($row->nick, $row->authid)) {
                $nick = $user_profile;
            } else {
                $nick = clean($row->nick, null);
            }

            $j = 0;
            $skill = (0 == $row->deaths)
                ? $row->frags
                : round($row->frags / $row->deaths, 2);

            do {
                $skill_name = $this->defaultSkillNames[$j];
                $skill_color = $this->skillsColors[$j];
                ++$j;
            } while (
                isset($this->defaultSkills[$j])
                && ($skill >= $this->defaultSkills[$j])
                && isset($this->defaultSkillNames[$j], $this->skillsColors[$j])
            );

            $tpl->load_template('elements/stat.tpl');
            $tpl->set('{suicide}', $row->suicide);
            $tpl->set('{skill_name}', $skill_name);
            $tpl->set('{skill_color}', $skill_color);
            $tpl->set('{skill}', $skill);
            $tpl->set('{kill_streak}', $row->kill_streak);
            $tpl->set('{death_streak}', $row->death_streak);
            $tpl->set('{teamkills}', $row->teamkills);
            $tpl->set('{time}', $row->gametime ? expand_seconds2($row->gametime) : '');
            $tpl->set('{type}', $server->st_type);
            $tpl->set('{id}', $row->id);
            $tpl->set('{place}', $row->place);
            $tpl->set('{nick}', $nick);
            $tpl->set('{frags}', $row->frags);
            $tpl->set('{deaths}', $row->deaths);
            $tpl->set('{headshots}', $row->headshots);
            $tpl->set('{shots}', $row->shots);
            $tpl->set('{hits}', $row->hits);
            $tpl->set('{kdr}', (0 == $row->deaths) ? $row->frags : round($row->frags / $row->deaths, 2));
            $tpl->set('{procent2}', get_procent($row->frags / 100, $row->headshots));
            $tpl->set('{procent3}', get_procent($row->frags / 100, $row->teamkills));
            $tpl->set('{procent4}', get_procent($row->shots / 100, $row->hits));
            $tpl->set('{date}', expand_date(date('d.m.Y H:i', $row->lasttime), 7));
            $tpl->set('{server}', $server->id);
            $tpl->set('{isHasRank}', $isHasRank);
            $tpl->compile('local_content');
            $tpl->clear();
        }

        if (empty($tpl->result['local_content'])) {
            return tableRowError('Нет результатов');
        }

        return $tpl->result['local_content'];
    }

    public function renderWeaponsStatistics(Server $server, \PDO $connection, \Template $tpl, string $playerId): string
    {
        return '';
    }

    public function renderMapsStatistics(Server $server, \PDO $connection, \Template $tpl, string $playerId): string
    {
        return '';
    }

    private function getSort($type): string
    {
        $sort = '';

        if (Integrations::SORT_BY_KILLS_DEATHS_TEAMKILLS_DIFF == $type) {
            $sort = 'CAST(kills AS SIGNED) - CAST(deaths AS SIGNED) - CAST(teamkills AS SIGNED) DESC';
        } elseif (Integrations::SORT_BY_KILLS == $type) {
            $sort = 'kills DESC';
        } elseif (Integrations::SORT_BY_KILLS_HEADSHODS_SUM == $type) {
            $sort = 'kills + headshots DESC';
        } elseif (Integrations::SORT_BY_SKILL == $type) {
            $sort = 'skill DESC';
        } elseif (Integrations::SORT_BY_ONLINE_TIME == $type) {
            $sort = 'connection_time DESC';
        }

        return $sort;
    }

    private function getWhere(
        $type,
        $frags = null,
        $deaths = null,
        $teamkills = null,
        $headshots = null,
        $skill = null,
        $gametime = null
    ): string {
        $where = '';

        if (Integrations::SORT_BY_KILLS_DEATHS_TEAMKILLS_DIFF == $type) {
            $where = '(CAST(kills AS SIGNED) - CAST(deaths AS SIGNED) - CAST(teamkills AS SIGNED)) > '.($frags - $deaths - $teamkills);
        } elseif (Integrations::SORT_BY_KILLS == $type) {
            $where = 'kills > '.$frags;
        } elseif (Integrations::SORT_BY_KILLS_HEADSHODS_SUM == $type) {
            $where = '(kills + headshots) > '.($frags + $headshots);
        } elseif (Integrations::SORT_BY_SKILL == $type) {
            $where = 'ROUND(skill, 3) > '.round($skill, 3);
        } elseif (Integrations::SORT_BY_ONLINE_TIME == $type) {
            $where = 'connection_time > '.$gametime;
        }

        return $where;
    }
}
