<?php

namespace GameCMS\Services\SteamApi;

use GameCMS\Common\Error;
use GameCMS\Services\SteamApi\DTO\AuthorizeResponseDTO;
use GameCMS\Services\SteamApi\Utils\LightOpenID;

class Authorization
{
    private string $host;
    private string $secret;
    private int $timeout;

    public function __construct(string $host, string $secret, int $timeout = 10)
    {
        $this->host = $host;
        $this->secret = $secret;
        $this->timeout = $timeout;
    }

    /**
     * @return array{?string, ?Error}
     */
    public function getAuthorizationUrl(string $redirectUrl, string $data = ''): array
    {
        $state = $this->generateCode(32);

        $_SESSION['steam_auth_state'] = $state;
        $_SESSION['steam_auth_data'] = $data;

        $url = null;
        $error = null;

        $redirectUrl .= (strpos($redirectUrl, '?') === false ? '?' : '&').'state='.$state;

        try {
            $openid = new LightOpenID($this->host);
            $openid->returnUrl = $redirectUrl;
            $openid->curl_time_out = $this->timeout;
            $openid->curl_connect_time_out = $this->timeout;
            $openid->identity = 'https://steamcommunity.com/openid';

            $url = $openid->authUrl();
        } catch (\Exception $e) {
            $error = error($e->getMessage());
        }

        return [$url, $error];
    }

    /**
     * @param mixed $incomeState
     *
     * @return array{?AuthorizeResponseDTO, ?Error}
     */
    public function authorize($incomeState): array
    {
        if (
            empty($_SESSION['steam_auth_state'])
            || empty($_SESSION['steam_auth_data'])
        ) {
            return [null, error('empty session data')];
        }

        if ($incomeState !== $_SESSION['steam_auth_state']) {
            return [null, error('invalid income state')];
        }

        try {
            $openid = new LightOpenID($this->host);

            if ($openid->mode == 'cancel') {
                return [null, error('mode is mode')];
            }

            if (!$openid->validate()) {
                return [null, error('validation error')];
            }
        } catch (\Exception $e) {
            return [null, error($e->getMessage())];
        }

        preg_match(
            '/^https:\\/\\/steamcommunity\\.com\\/openid\\/id\\/(7[0-9]{15,25}+)$/',
            $openid->__get('identity'),
            $matches
        );

        if (empty($matches[1])) {
            return [null, error('invalid steam id')];
        }

        $steamId = $matches[1];
        $data = $_SESSION['steam_auth_data'];

        $usersService = new Users($this->secret, $this->timeout);
        [$users, $error] = $usersService->getInfo([$steamId]);
        if ($error) {
            return [null, error($error)];
        }

        if (!$users || !$users[0]) {
            return [null, error('cannot get user info')];
        }

        $userInfo = $users[0];

        $response = new AuthorizeResponseDTO();
        $response->steamId = $steamId;
        $response->login = $userInfo->login;
        $response->avatar = $userInfo->avatar;
        $response->data = $data;

        return [$response, null];
    }

    private function generateCode(int $length = 128): string
    {
        $chars = 'qazxswedcvfrtgbnhyujmkiolp1234567890QAZXSWEDCVFRTGBNHYUJMKIOLP';

        $code = '';
        while ($length--) {
            $code .= $chars[rand(0, strlen($chars) - 1)];
        }

        return $code;
    }
}
