<?php

namespace GameCMS\Services;

use GameCMS\Common\Error;
use GameCMS\Integrations\Admins\AdminSystemIntegration;
use GameCMS\Integrations\Admins\AmxBansIntegration;
use GameCMS\Integrations\Admins\BlankIntegration;
use GameCMS\Integrations\Admins\GameCMSIntegration;
use GameCMS\Integrations\Admins\IksIntegration;
use GameCMS\Integrations\Admins\SourceBansIntegration;
use GameCMS\Integrations\Admins\UsersFileIntegration;
use GameCMS\Integrations\Integrations;
use GameCMS\Repositories\AdminRepository;
use GameCMS\Repositories\ServerRepository;
use GameCMS\Utils\Lock;

class AdminsService
{
    public function getIntegrator(int $integration): ?AdminSystemIntegration
    {
        switch ($integration) {
            case Integrations::USERS_FILE:
            case Integrations::AMXBANS_WITH_USERS_FILE:
                return new UsersFileIntegration();

            case Integrations::AMXBANS:
                return new AmxBansIntegration();

            case Integrations::SOURCEBANS:
                return new SourceBansIntegration();

            case Integrations::IKS:
                return new IksIntegration();

            case Integrations::AMXBANS_WITH_GAMECMS_API:
                return new GameCMSIntegration();

            case Integrations::BLANK:
            default:
                return new BlankIntegration();
        }
    }

    /**
     * @return array{bool, ?Error}
     */
    public function isIntegratorStorageAvailable(int $server_id): array
    {
        $server = ServerRepository::getById($server_id);
        if (empty($server)) {
            return [false, error('Сервер не найден')];
        }

        $integrator = $this->getIntegrator($server->type);
        if (empty($integrator)) {
            return [false, error('Не найден способ интеграции сервера')];
        }

        [, $error] = $integrator->getStorageConnection($server);
        if ($error) {
            return [false, $error];
        }

        return [true, null];
    }

    /**
     * @return array{bool, ?Error}
     */
    public function syncAdminWithIntegratorStorage(int $adminId, ?string $nameForSearch = null): array
    {
        $server = ServerRepository::getByAdminId($adminId);
        if (empty($server)) {
            return [false, error('Сервер не найден')];
        }

        $integrator = $this->getIntegrator($server->type);
        if (empty($integrator)) {
            return [false, error('Не найден способ интеграции сервера')];
        }

        $admin = AdminRepository::getById($adminId);
        if (empty($admin)) {
            return [false, error('Админ не найден')];
        }

        if (1 !== $admin->active) {
            return [false, error('Администратор выключен')];
        }

        if (0 !== $admin->pause) {
            return [false, error('Администратор на паузе')];
        }

        if (!$nameForSearch) {
            $nameForSearch = $admin->name;
        }

        [$connection, $error] = $integrator->getStorageConnection($server);
        if ($error) {
            return [false, $error];
        }

        [$isSuccess, $error] = $integrator->exportAdmin($nameForSearch, $connection, $server, $admin);
        if (!$isSuccess) {
            return [false, $error];
        }

        $integrator->notyServer($server);

        $this->updateAdminRemoveTime();

        return [true, null];
    }

    /**
     * @return array{bool, ?Error}
     */
    public function dellAdminFromIntegratorStorage(int $adminId): array
    {
        $server = ServerRepository::getByAdminId($adminId);
        if (empty($server)) {
            return [false, error('Сервер не найден')];
        }

        $integrator = $this->getIntegrator($server->type);
        if (empty($integrator)) {
            return [false, error('Не найден способ интеграции сервера')];
        }

        $admin = AdminRepository::getById($adminId);
        if (empty($admin)) {
            return [false, error('Админ не найден')];
        }

        [$connection, $error] = $integrator->getStorageConnection($server);
        if ($error) {
            return [false, $error];
        }

        [$isSuccess, $error] = $integrator->removeAdmin($admin->name, $connection, $server);
        if (!$isSuccess) {
            return [false, $error];
        }

        $integrator->notyServer($server);

        $this->updateAdminRemoveTime();

        return [true, null];
    }

    /**
     * @return array{bool, ?Error}
     */
    public function dellAdminAndFromSiteAndIntegratorStorage(int $adminId): array
    {
        [$isSuccess, $error] = $this->dellAdminFromIntegratorStorage($adminId);
        if (!$isSuccess && !$error->is('iks_remove_admin_not_found')) {
            return [false, $error];
        }

        AdminRepository::removeById($adminId);

        return [true, null];
    }

    /**
     * @return array{bool, ?Error}
     */
    public function importAdminsFromIntegratorStorage(int $serverId): array
    {
        $server = ServerRepository::getById($serverId);
        if (empty($server)) {
            return [false, error('Сервер не найден')];
        }

        $integrator = $this->getIntegrator($server->type);
        if (empty($integrator)) {
            return [false, error('Не найден способ интеграции сервера')];
        }

        [$connection, $error] = $integrator->getStorageConnection($server);
        if ($error) {
            return [false, $error];
        }

        AdminRepository::removeByServerId($server->id);

        [$isSuccess, $error] = $integrator->importAdmins($connection, $server);
        if (!$isSuccess) {
            return [false, $error];
        }

        $this->updateAdminRemoveTime();

        return [true, null];
    }

    /**
     * @return array{bool, ?Error}
     */
    public function exportAdminsToIntegratorStorage(int $serverId): array
    {
        $server = ServerRepository::getById($serverId);
        if (empty($server)) {
            return [false, error('Сервер не найден')];
        }

        $integrator = $this->getIntegrator($server->type);
        if (empty($integrator)) {
            return [false, error('Не найден способ интеграции сервера')];
        }

        [$connection, $error] = $integrator->getStorageConnection($server);
        if ($error) {
            return [false, $error];
        }

        $admins = AdminRepository::getActiveByServerId($server->id);

        [$isSuccess, $error] = $integrator->exportAdmins($connection, $server, $admins);
        if (!$isSuccess) {
            return [false, $error];
        }

        $integrator->notyServer($server);

        return [true, null];
    }

    public function updateAdminGroup(int $user_id, int $admin_id, int $admin_service_id): void
    {
        if (0 != $admin_id) {
            $STH = pdo()->prepare(
                "SELECT services.users_group FROM admins__services 
				LEFT JOIN admins ON admins.id = admins__services.admin_id 
				LEFT JOIN services ON services.id = admins__services.service
				WHERE admins.user_id=:user_id AND admins__services.previous_group!='0' AND admins__services.admin_id!=:admin_id ORDER BY admins__services.bought_date DESC LIMIT 1"
            );
            $STH->setFetchMode(\PDO::FETCH_OBJ);
            $STH->execute([':admin_id' => $admin_id, ':user_id' => $user_id]);
            $row = $STH->fetch();

            if (empty($row->users_group)) {
                $STH = pdo()->prepare(
                    "SELECT previous_group FROM admins__services WHERE previous_group!='0' AND admin_id=:admin_id LIMIT 1"
                );
                $STH->setFetchMode(\PDO::FETCH_OBJ);
                $STH->execute([':admin_id' => $admin_id]);
                $row = $STH->fetch();

                if (!empty($row->previous_group)) {
                    $previous_group = $row->previous_group;
                } else {
                    $previous_group = 0;
                }
            } else {
                $previous_group = $row->users_group;
            }
        } elseif (0 != $admin_service_id) {
            $STH = pdo()->prepare(
                "SELECT services.users_group FROM admins__services 
				LEFT JOIN admins ON admins.id = admins__services.admin_id 
				LEFT JOIN services ON services.id = admins__services.service
				WHERE admins.user_id=:user_id AND admins__services.previous_group!='0' AND admins__services.id!=:admin_service_id ORDER BY admins__services.bought_date DESC LIMIT 1"
            );
            $STH->setFetchMode(\PDO::FETCH_OBJ);
            $STH->execute([':admin_service_id' => $admin_service_id, ':user_id' => $user_id]);
            $row = $STH->fetch();

            if (empty($row->users_group)) {
                $STH = pdo()->prepare(
                    "SELECT previous_group FROM admins__services WHERE previous_group!='0' AND id=:admin_service_id LIMIT 1"
                );
                $STH->setFetchMode(\PDO::FETCH_OBJ);
                $STH->execute([':admin_service_id' => $admin_service_id]);
                $row = $STH->fetch();

                if (!empty($row->previous_group)) {
                    $previous_group = $row->previous_group;
                } else {
                    $previous_group = 0;
                }
            } else {
                $previous_group = $row->users_group;
            }
        }

        if (0 != $previous_group) {
            $STH = pdo()->prepare('UPDATE users SET rights=:rights WHERE id=:id LIMIT 1');
            $STH->execute([':rights' => $previous_group, ':id' => $user_id]);
        }
    }

    public function notyAdminsAboutExpire(): void
    {
        incNotifications();

        $now = date('Y-m-d H:i:s');
        $date = date('Y-m-d H:i:s', strtotime($now) + 3600 * 24 * 5);

        $STH = pdo()->prepare(
            "SELECT 
    							admins__services.id,
							    users.email,
							    users.email_notice,
							    servers.name AS server_name,
							    servers.type AS server_type,
							    admins__services.admin_id,
							    admins__services.ending_date,
							    admins.server,
							    admins.name,
							    admins.user_id,
							    services.name AS service_name
							FROM 
			 				    admins__services 
							LEFT JOIN 
			 					admins 
			 				ON 
			 					admins.id = admins__services.admin_id 
							LEFT JOIN 
			 				    services 
			 				ON 
			 				    services.id = admins__services.service 
							LEFT JOIN 
			 				    servers 
			 				ON
			 					servers.id = admins.server 
							LEFT JOIN 
			 				    users 
			 				ON 
			 					users.id = admins.user_id 
							WHERE 
							    admins__services.ending_date!='0000-00-00 00:00:00' 
							  	AND admins__services.ending_date<:date 
							  	AND admins.active = '1' 
								AND admins.pause = 0 
								AND servers.type != 0"
        );
        $STH->execute([':date' => $date]);
        while ($admin = $STH->fetchObject()) {
            if (empty($admin->user_id)) {
                continue;
            }

            $left = strtotime($admin->ending_date) - time();
            $left = expand_seconds2($left, 2);

            if (empty($admin->service_name)) {
                $admin->service_name = 'Неизвестно';
            }

            $noty = noty_of_ending_service(
                $left,
                $admin->name,
                $admin->service_name,
                $admin->server_name
            );
            send_noty($noty, $admin->user_id, 3);

            if (
                !empty($admin->email)
                && 'vk_id_' != substr($admin->email, 6)
                && 1 == $admin->email_notice
            ) {
                $letter = letter_of_ending_service(
                    configs()->name,
                    $left,
                    $admin->name,
                    $admin->service_name,
                    $admin->server_name
                );
                sendmail($admin->email, $letter['subject'], $letter['message'], pdo());
            }
        }
    }

    public function removeExpiredAdmins(): bool
    {
        incNotifications();

        $now = date('Y-m-d H:i:s');

        if (configs()->dell_admin_time > $now) {
            return true;
        }

        $lock = new Lock('remove-expired-admins');
        if (!$lock->lock()) {
            return true;
        }

        $STH = pdo()->prepare(
            "SELECT 
					    services.users_group,
					    admins__services.id,
    					admins__services.service,
					    users.email,
					    users.email_notice,
					    servers.name AS server_name,
					    servers.type AS server_type,
					    admins__services.admin_id,
					    admins__services.ending_date,
					    admins.server,
					    admins.name,
					    admins.user_id,
					    services.name AS service_name
			 		FROM 
			 		    admins__services 
					LEFT JOIN 
			 			admins 
			 		ON 
			 		    admins.id = admins__services.admin_id 
					LEFT JOIN 
			 			services 
			 		ON 
			 		    services.id = admins__services.service 
					LEFT JOIN 
			 			servers 
			 		ON 
			 		    servers.id = admins.server 
					LEFT JOIN 
			 			users 
			 		ON 
			 		    users.id = admins.user_id 
					WHERE 
					    admins__services.ending_date!=:ending_date 
						AND admins__services.ending_date<:now 
						AND admins.active = '1' 
						AND admins.pause = '0' 
						AND servers.type != 0"
        );
        $STH->execute([':ending_date' => '0000-00-00 00:00:00', ':now' => $now]);
        while ($admin = $STH->fetchObject()) {
            $STH2 = pdo()->prepare('SELECT id FROM admins__services WHERE admin_id=:id ');
            $STH2->execute([':id' => $admin->admin_id]);
            $row = $STH2->fetchAll();
            $count = count($row);

            if (1 == $count) {
                service_log('Автоматическое полное удаление прав', $admin->admin_id, $admin->server);

                if (!empty($admin->user_id)) {
                    $this->updateAdminGroup($admin->user_id, $admin->admin_id, 0);
                }

                [$isSuccess, $error] = $this->dellAdminAndFromSiteAndIntegratorStorage($admin->admin_id);
                if (!$isSuccess) {
                    log_error("dell_old_admins: {$error}");

                    return false;
                }
            } else {
                service_log(
                    'Автоматическое удаление услуги',
                    $admin->admin_id,
                    $admin->server,
                    $admin->service
                );

                if (!empty($admin->user_id)) {
                    $this->updateAdminGroup($admin->user_id, 0, $admin->id);
                }

                $STH2 = pdo()->prepare('DELETE FROM admins__services WHERE id=:id LIMIT 1');
                $STH2->execute([':id' => $admin->id]);

                [$isSuccess, $error] = $this->syncAdminWithIntegratorStorage($admin->admin_id);
                if (!$isSuccess) {
                    log_error("dell_admin_service: {$error}");

                    return false;
                }
            }

            if (!empty($admin->user_id)) {
                if (empty($admin->service_name)) {
                    $admin->service_name = 'Неизвестно';
                }

                $noty = noty_of_dell_service($admin->name, $admin->service_name, $admin->server_name);
                send_noty($noty, $admin->user_id, 3);

                if (!empty($admin->email) && 1 == $admin->email_notice) {
                    $letter = letter_of_dell_service(
                        configs()->name,
                        $admin->name,
                        $admin->service_name,
                        $admin->server_name
                    );
                    sendmail($admin->email, $letter['subject'], $letter['message'], pdo());
                }
            }
        }

        $this->updateAdminRemoveTime();

        $lock->unLock();
        $lock->close();

        return true;
    }

    private function updateAdminRemoveTime(): void
    {
        $STH = pdo()->query(
            "SELECT 
							    admins__services.ending_date 
							FROM 
							    admins__services 
							LEFT JOIN 
								admins 
							ON 
								admins.id = admins__services.admin_id 
							LEFT JOIN 
							        servers 
							ON 
							    servers.id = admins.server 
							WHERE 
							    admins__services.ending_date != '0000-00-00 00:00:00' 
								AND admins.active = '1' 
								AND admins.pause = '0' 
								AND servers.type != 0
							ORDER BY 
							    admins__services.ending_date
							LIMIT 1"
        );
        $STH->setFetchMode(\PDO::FETCH_OBJ);
        $row = $STH->fetch();
        if (empty($row->ending_date)) {
            $ending_date = date('Y-m-d H:i:s', strtotime(date('Y-m-d H:i:s')) + 3600 * 3);
        } else {
            $ending_date = $row->ending_date;
        }

        $STH = pdo()->prepare('UPDATE config SET dell_admin_time=:dell_admin_time LIMIT 1');
        $STH->execute([':dell_admin_time' => $ending_date]);
    }
}
