<?php

namespace GameCMS\Integrations\Admins;

use GameCMS\Models\Admin;
use GameCMS\Models\Server;

class AmxBansIntegration extends DatabasedIntegration implements AdminSystemIntegration
{
    use AdminSystemHelpers;

    public function isBlank(): bool
    {
        return false;
    }

    public function removeAdmin(string $adminFindName, $connection, Server $server): array
    {
        $adminId = $this->getAdminIdFromStorage(
            $connection,
            $adminFindName,
            $server->db_prefix,
            $server->ip,
            $server->port
        );

        if (!$adminId) {
            return [false, error('Админ не найден в базе сервера')];
        }

        $table = set_prefix($server->db_prefix, 'amxadmins');
        $STH = $connection->prepare("DELETE FROM `{$table}` WHERE `id`=:id LIMIT 1");
        $STH->execute([':id' => $adminId]);

        $table = set_prefix($server->db_prefix, 'admins_servers');
        $STH = $connection->prepare("DELETE FROM `{$table}` WHERE `admin_id`=:id LIMIT 1");
        $STH->execute([':id' => $adminId]);

        return [true, null];
    }

    public function exportAdmin(string $adminFindName, $connection, Server $server, Admin $admin): array
    {
        $adminServices = $this->getAdminServices($admin->id);
        $rights = $this->getAdminRights($adminServices);
        $rights['flags'] = $this->collectAdminRights($rights['flags']);
        $rights['expired'] = $this->collectAdminExpireDate($adminServices);
        $adminServices = $this->serializeAdminInfo($adminServices);

        $admin->name = htmlspecialchars_decode($admin->name, ENT_QUOTES);

        if (empty($admin->pass_md5)) {
            $admin->pass_md5 = null;
        }

        $adminId = $this->getAdminIdFromStorage(
            $connection,
            $adminFindName,
            $server->db_prefix,
            $server->ip,
            $server->port
        );

        $table = set_prefix($server->db_prefix, 'amxadmins');
        if ($adminId) {
            $STH = $connection->prepare(
                "UPDATE `{$table}` SET `username`=:username, `nickname`=:nickname, `password`=:password, `access`=:access, `flags`=:flags, `steamid`=:steamid, `created`=:created, `expired`=:expired, `days`=:days, `gamecms`=:gamecms WHERE `id`=:id LIMIT 1"
            );
            $STH->execute(
                [
                    ':username' => $admin->user_id,
                    ':nickname' => $admin->name,
                    ':password' => $admin->pass_md5,
                    ':access' => $rights['flags'],
                    ':flags' => $admin->type,
                    ':steamid' => $admin->name,
                    ':created' => '0',
                    ':expired' => $rights['expired'],
                    ':days' => '0',
                    ':gamecms' => $adminServices,
                    ':id' => $adminId,
                ]
            );
        } else {
            $connection
                ->prepare(
                    "INSERT INTO `{$table}` (`username`,`nickname`,`password`,`access`,`flags`,`steamid`,`created`,`expired`,`days`,`gamecms`) values (:username, :nickname, :password, :access, :flags, :steamid, :created, :expired, :days, :gamecms)"
                )
                ->execute(
                    [
                        ':username' => $admin->user_id,
                        ':nickname' => $admin->name,
                        ':password' => $admin->pass_md5,
                        ':access' => $rights['flags'],
                        ':flags' => $admin->type,
                        ':steamid' => $admin->name,
                        ':created' => '0',
                        ':expired' => $rights['expired'],
                        ':days' => '0',
                        ':gamecms' => $adminServices,
                    ]
                );

            $table = set_prefix($server->db_prefix, 'serverinfo');
            $STH = $connection->prepare("SELECT `id` FROM `{$table}` WHERE `address`=:address LIMIT 1");
            $STH->setFetchMode(\PDO::FETCH_OBJ);
            $STH->execute([':address' => $server->ip.':'.$server->port]);
            $serverInfo = $STH->fetch();

            $amxAdminId = get_ai($connection, set_prefix($server->db_prefix, 'amxadmins')) - 1;
            $table = set_prefix($server->db_prefix, 'admins_servers');

            $connection
                ->prepare(
                    "INSERT INTO `{$table}` (`admin_id`,`server_id`,`use_static_bantime`,`custom_flags`) values (:admin_id, :server_id, :use_static_bantime, :custom_flags)"
                )
                ->execute(
                    [
                        ':admin_id' => $amxAdminId,
                        ':server_id' => $serverInfo->id,
                        ':use_static_bantime' => 'no',
                        'custom_flags' => '',
                    ]
                );
        }

        return [true, null];
    }

    public function importAdmins($connection, Server $server): array
    {
        $table = set_prefix($server->db_prefix, 'serverinfo');

        $STH = $connection->prepare("SELECT id FROM {$table} WHERE address=:address LIMIT 1");
        $STH->execute([':address' => $server->ip.':'.$server->port]);
        $row = $STH->fetchObject();
        if (empty($row->id)) {
            return [false, error('Сервер не найден')];
        }

        $adminsServersTable = set_prefix($server->db_prefix, 'admins_servers');
        $adminsTable = set_prefix($server->db_prefix, 'amxadmins');
        $STH = $connection->query(
            "SELECT 
					{$adminsServersTable}.admin_id,
					{$adminsTable}.* 
				FROM {$adminsServersTable} 
					LEFT JOIN {$adminsTable} ON {$adminsTable}.id = {$adminsServersTable}.admin_id 
				WHERE {$adminsServersTable}.server_id = {$row->id}"
        );
        $STH->execute();
        $admins = $STH->fetchAll(\PDO::FETCH_OBJ);
        foreach ($admins as $serverAdmin) {
            if (empty($serverAdmin->steamid)) {
                continue;
            }

            $siteAdmin = [];
            $services = [];

            $siteAdmin['name'] = check($serverAdmin->steamid);
            $siteAdmin['type'] = check($serverAdmin->flags);
            $siteAdmin['pass'] = '';
            $siteAdmin['pass_md5'] = '';
            $siteAdmin['user_id'] = 0;

            if (!empty($serverAdmin->password)) {
                $siteAdmin['pass_md5'] = $serverAdmin->password;
            }

            if (!empty($serverAdmin->username)) {
                $siteAdmin['user_id'] = check($serverAdmin->username, 'int');
            }

            if (!empty($serverAdmin->gamecms)) {
                $services = unserialize($serverAdmin->gamecms);
            } else {
                if (0 == $serverAdmin->expired || 0 == $serverAdmin->created) {
                    $serviceTime = 0;
                } else {
                    $serviceTime = round(($serverAdmin->expired - $serverAdmin->created) / 60 / 60 / 24);
                }

                $endingDate = '0000-00-00 00:00:00';
                if (0 != $serverAdmin->expired) {
                    $endingDate = date('Y-m-d H:i:s', $serverAdmin->expired);
                }

                $boughtDate = '0000-00-00 00:00:00';
                if (0 != $serverAdmin->created) {
                    $boughtDate = date('Y-m-d H:i:s', $serverAdmin->created);
                }

                $services[] = [
                    'service' => 0,
                    'rights_und' => check($serverAdmin->access),
                    'service_time' => $serviceTime,
                    'bought_date' => $boughtDate,
                    'ending_date' => $endingDate,
                    'irretrievable' => 0,
                ];
            }

            $this->insertAdmin($server->id, $siteAdmin, $services);
        }

        return [true, null];
    }

    public function exportAdmins($connection, Server $server, array $admins): array
    {
        foreach ($admins as $admin) {
            [$isSuccess, $error] = $this->exportAdmin($admin->name, $connection, $server, $admin);
            if (!$isSuccess) {
                return [false, $error];
            }
        }

        return [true, null];
    }

    public function isImmunityUsing(): bool
    {
        return false;
    }

    public function isGroupsUsing(): bool
    {
        return false;
    }

    public function isNickWithPasswordAuthAllowed(): bool
    {
        return true;
    }

    public function isSteamIdAuthAllowed(): bool
    {
        return true;
    }

    public function isSteamIdWithPasswordAuthAllowed(): bool
    {
        return true;
    }

    private function getAdminIdFromStorage(
        \PDO $connection,
        string $adminFindName,
        string $dbPrefix,
        string $ip,
        int $port
    ): ?int {
        $adminFindName = htmlspecialchars_decode($adminFindName, ENT_QUOTES);

        $table = set_prefix($dbPrefix, 'serverinfo');
        $STH = $connection->prepare("SELECT `id` FROM `{$table}` WHERE `address`=:address LIMIT 1");
        $STH->execute([':address' => $ip.':'.$port]);
        $serverInfo = $STH->fetchObject();
        if (empty($serverInfo)) {
            return 0;
        }

        $table = set_prefix($dbPrefix, 'amxadmins');
        $STH = $connection->prepare("SELECT `id` FROM `{$table}` WHERE `steamid`=:steamid");
        $STH->execute([':steamid' => $adminFindName]);
        $amxAdmin = $STH->fetchAll();
        $count = count($amxAdmin);

        $table = set_prefix($dbPrefix, 'admins_servers');
        for ($i = 0; $i < $count; ++$i) {
            $STH = $connection->prepare(
                "SELECT `admin_id` FROM `{$table}` WHERE `admin_id`=:admin_id AND `server_id`=:server_id LIMIT 1"
            );
            $STH->execute([':admin_id' => $amxAdmin[$i]['id'], ':server_id' => $serverInfo->id]);
            $row = $STH->fetchObject();
            if (isset($row->admin_id)) {
                return $amxAdmin[$i]['id'];
            }
        }

        return 0;
    }
}
