<?php

namespace GameCMS\Statistics;

use GameCMS\Models\Server;

class CsStatsMysqlIntegration extends DatabasedIntegration implements StatisticIntegration
{
    protected \GetData $getData;

    protected string $whereStatement = '(frags!=0)';

    protected array $sortTypes = [
        Integrations::SORT_BY_KILLS_DEATHS_TEAMKILLS_DIFF,
        Integrations::SORT_BY_KILLS,
        Integrations::SORT_BY_KILLS_HEADSHODS_SUM,
        Integrations::SORT_BY_SKILL,
        Integrations::SORT_BY_ONLINE_TIME,
        Integrations::SORT_BY_PLACE,
        Integrations::SORT_BY_ADVANCED,
    ];

    protected array $defaultSkills = [0, 60, 75, 85, 100, 115, 130, 140, 150, 165, 180, 195, 210];

    protected array $defaultSkillNames = ['L-', 'L', 'L+', 'M-', 'M', 'M+', 'H-', 'H', 'H+', 'P-', 'P', 'P+', 'G'];

    protected array $skillsColors = [
        'default',
        'default',
        'default',
        'success',
        'success',
        'success',
        'primary',
        'primary',
        'primary',
        'warning',
        'warning',
        'warning',
        'danger',
    ];

    public function __construct()
    {
        $this->getData = new \GetData();
    }

    public function isBlank(): bool
    {
        return false;
    }

    public function isValidStatisticsSortType(int $type): bool
    {
        return in_array($type, $this->sortTypes);
    }

    public function isValidDataBase(\PDO $connection, string $table = ''): bool
    {
        if (!check_table('csstats_players', $connection) || !check_table('csstats_settings', $connection)) {
            return false;
        }

        return true;
    }

    public function removeStats(Server $server, \PDO $connection, string $playerId): bool
    {
        $STH = $connection->prepare('DELETE FROM csstats_players WHERE id=:id LIMIT 1');
        $STH->execute([':id' => $playerId]);

        return true;
    }

    public function getStatsCount(Server $server, \PDO $connection): int
    {
        return $connection
            ->query("SELECT COUNT(*) as count FROM csstats_players WHERE {$this->whereStatement}")
            ->fetchColumn();
    }

    public function isHasRank(Server $server, \PDO $connection): bool
    {
        return false;
    }

    public function renderCommonStatistics(
        Server $server,
        \PDO $connection,
        \Template $tpl,
        string $name,
        int $start,
        int $limit
    ): string {
        $tpl->result['local_content'] = '';

        $isHasRank = $this->isHasRank($server, $connection);

        $STH = $connection->query('SELECT * FROM csstats_settings');
        while ($row = $STH->fetchObject()) {
            if ('statsx_skill' == $row->command) {
                $skills = explode(' ', $row->value);
            } elseif ('statsx_skillname' == $row->command) {
                $skills_names = explode(' ', $row->value);
            }
        }

        if (empty($skills[1])) {
            $skills = $this->defaultSkills;
        }

        if (empty($skills_names[1])) {
            $skills_names = $this->defaultSkillNames;
        }

        $sort = $this->getSort($server->st_sort_type);

        if (empty($name)) {
            $STH = $connection->query(
                "SELECT *
				FROM csstats_players
				WHERE {$this->whereStatement}
				ORDER BY {$sort}
				LIMIT {$start}, {$limit}"
            );
        } else {
            $STH = $connection->prepare(
                prepareSearchQuery(
                    "SELECT 
						*
					FROM csstats_players
					WHERE {$this->whereStatement}
					  AND (nick LIKE :name OR authid LIKE :name OR ip LIKE :name)
					ORDER BY {$sort}"
                )
            );
            $STH->execute([':name' => getNameLike($name)]);
        }

        while ($row = $STH->fetchObject()) {
            ++$start;

            if (empty($row->authid)) {
                $row->authid = '';
            }

            if (empty($name)) {
                $row->place = $start;
            } else {
                $where = $this->getWhere(
                    $server->st_sort_type,
                    $row->frags,
                    $row->deaths,
                    $row->teamkills,
                    $row->headshots,
                    $row->skill,
                    $row->gametime,
                    $row->suicide,
                    $row->defused,
                    $row->explode,
                    $row->place
                );

                $STH2 = $connection->query(
                    "SELECT 
						count(*)
					FROM csstats_players
					WHERE {$this->whereStatement} 
						AND ({$where})
					ORDER BY {$sort}"
                );
                $row->place = $STH2->fetchColumn() + 1;
            }

            if ($user_profile = $this->getData->get_gamer_profile($row->nick, $row->authid)) {
                $nick = $user_profile;
            } else {
                $nick = clean($row->nick, null);
            }

            $j = 0;
            do {
                $skill_name = $skills_names[$j];
                $skill_color = $this->skillsColors[$j];
                ++$j;
            } while (isset($skills[$j]) && ($row->skill >= $skills[$j]) && isset($skills_names[$j], $this->skillsColors[$j]));

            $tpl->load_template('elements/stat.tpl');
            $tpl->set('{suicide}', $row->suicide);
            $tpl->set('{skill_name}', $skill_name);
            $tpl->set('{skill_color}', $skill_color);
            $tpl->set('{skill}', round($row->skill));
            $tpl->set('{damage}', $row->damage);
            $tpl->set('{defusing}', $row->defusing);
            $tpl->set('{connects}', $row->connects);
            $tpl->set('{authid}', isNeedHidePlayerId() ? hidePlayerId($row->authid) : $row->authid);
            $tpl->set('{authid_original}', $row->authid);
            $tpl->set('{defused}', $row->defused);
            $tpl->set('{planted}', $row->planted);
            $tpl->set('{explode}', $row->explode);
            $tpl->set('{rounds}', $row->rounds);
            $tpl->set('{wint}', $row->wint);
            $tpl->set('{winct}', $row->winct);
            $tpl->set('{teamkills}', $row->teamkills);
            $tpl->set('{time}', $row->gametime ? expand_seconds2($row->gametime) : '');
            $tpl->set('{type}', $server->st_type);
            $tpl->set('{isHasRank}', $isHasRank);
            $tpl->set('{id}', $row->id);
            $tpl->set('{place}', $row->place);
            $tpl->set('{nick}', $nick);
            $tpl->set('{frags}', $row->frags);
            $tpl->set('{deaths}', $row->deaths);
            $tpl->set('{headshots}', $row->headshots);
            $tpl->set('{shots}', $row->shots);
            $tpl->set('{hits}', $row->hits);
            $tpl->set('{kdr}', (0 == $row->deaths) ? $row->frags : round($row->frags / $row->deaths, 2));
            $tpl->set('{procent2}', get_procent($row->frags / 100, $row->headshots));
            $tpl->set('{procent3}', get_procent($row->frags / 100, $row->teamkills));
            $tpl->set('{procent4}', get_procent($row->shots / 100, $row->hits));
            $tpl->set('{date}', expand_date(date('d.m.Y H:i', $row->lasttime), 7));
            $tpl->set('{server}', $server->id);
            $tpl->compile('local_content');
            $tpl->clear();
        }

        if (empty($tpl->result['local_content'])) {
            return tableRowError('Нет результатов');
        }

        return $tpl->result['local_content'];
    }

    public function renderWeaponsStatistics(Server $server, \PDO $connection, \Template $tpl, string $playerId): string
    {
        return '';
    }

    public function renderMapsStatistics(Server $server, \PDO $connection, \Template $tpl, string $playerId): string
    {
        return '';
    }

    protected function getSort($type): string
    {
        $sort = '';

        if (Integrations::SORT_BY_KILLS_DEATHS_TEAMKILLS_DIFF == $type) {
            $sort = 'frags - deaths - teamkills DESC';
        } elseif (Integrations::SORT_BY_KILLS == $type) {
            $sort = 'frags DESC';
        } elseif (Integrations::SORT_BY_KILLS_HEADSHODS_SUM == $type) {
            $sort = 'frags + headshots DESC';
        } elseif (Integrations::SORT_BY_SKILL == $type) {
            $sort = 'skill DESC';
        } elseif (Integrations::SORT_BY_ONLINE_TIME == $type) {
            $sort = 'gametime DESC';
        } elseif (Integrations::SORT_BY_PLACE == $type) {
            $sort = 'place';
        } elseif (Integrations::SORT_BY_ADVANCED == $type) {
            $sort = 'frags - deaths + headshots - teamkills * 2 -suicide * 3 + defused * 3 + explode * 3 DESC';
        }

        return $sort;
    }

    protected function getWhere(
        $type,
        $frags = null,
        $deaths = null,
        $teamkills = null,
        $headshots = null,
        $skill = null,
        $gametime = null,
        $suicide = null,
        $defused = null,
        $explode = null,
        $place = null
    ): string {
        $where = '';

        if (Integrations::SORT_BY_KILLS_DEATHS_TEAMKILLS_DIFF == $type) {
            $where = '(frags - deaths - teamkills) > '.($frags - $deaths - $teamkills);
        } elseif (Integrations::SORT_BY_KILLS == $type) {
            $where = 'frags > '.$frags;
        } elseif (Integrations::SORT_BY_KILLS_HEADSHODS_SUM == $type) {
            $where = '(frags + headshots) > '.($frags + $headshots);
        } elseif (Integrations::SORT_BY_SKILL == $type) {
            $where = 'ROUND(skill, 3) > '.round($skill, 3);
        } elseif (Integrations::SORT_BY_ONLINE_TIME == $type) {
            $where = 'gametime > '.$gametime;
        } elseif (Integrations::SORT_BY_PLACE == $type) {
            $where = 'place < '.$place;
        } elseif (Integrations::SORT_BY_ADVANCED == $type) {
            $where = '(frags - deaths + headshots - teamkills * 2 -suicide * 3 + defused * 3 + explode * 3) > '
                .($frags - $deaths + $headshots - $teamkills * 2 - $suicide * 3 + $defused * 3 + $explode * 3);
        }

        return $where;
    }
}
