<?php

namespace GameCMS\Integrations\Admins;

use GameCMS\Common\Error;
use GameCMS\Models\Admin;
use GameCMS\Models\Server;
use GameCMS\Repositories\AdminRepository;

class UsersFileIntegration implements AdminSystemIntegration
{
    use AdminSystemHelpers;

    public function isBlank(): bool
    {
        return false;
    }

    public function getStorageConnection(Server $server): array
    {
        $connection = \FtpConnection::connection(
            $server->ftp_host,
            $server->ftp_port,
            $server->ftp_login,
            $server->ftp_pass,
            self::class
        );

        if (!$connection) {
            return [null, error('Не удалось подключиться к FTP сервера')];
        }

        return [$connection, null];
    }

    public function exportAdmin(string $adminFindName, $connection, Server $server, Admin $admin): array
    {
        $admins = AdminRepository::getActiveByServerId($server->id);

        return $this->export($connection, $server, $admins);
    }

    public function removeAdmin(string $adminFindName, $connection, Server $server): array
    {
        $admins = AdminRepository::getActiveByServerId($server->id);

        return $this->export(
            $connection,
            $server,
            array_filter(
                $admins,
                function ($admin) use ($adminFindName) {
                    return $admin->name !== $adminFindName;
                }
            )
        );
    }

    public function importAdmins($connection, Server $server): array
    {
        $tempFile = __DIR__.'/../../../../../files/temp/users'.rand().'.txt';

        if (!$file = fopen($tempFile, 'w')) {
            return [false, error('Не удалось создать временный файл')];
        }

        if (!ftp_fget($connection, $file, $server->ftp_string, FTP_ASCII, 0)) {
            return [false, error('Не удалось получить файл с FTP сервера')];
        }

        if (!$admins = file_get_contents($tempFile)) {
            return [false, error('Не удалось создать временный файл')];
        }

        AdminRepository::removeByServerId($server->id);

        $adminsRows = explode(':end:', $admins);
        foreach ($adminsRows as $adminRow) {
            $data = explode(';', trim($adminRow), 2);

            if (isset($data['0'], $data['1'])) {
                $adminParams = explode('"', $data['0']);
                $serviceParams = explode("'", $data['1']);
            } else {
                continue;
            }

            if (
                empty($adminParams['1'])
                || empty($adminParams['5'])
                || empty($adminParams['7'])
                || (';' == trim($adminParams['0']))
            ) {
                continue;
            }

            $siteAdmin = [];
            $services = [];

            $siteAdmin['name'] = check($adminParams['1']);
            $siteAdmin['type'] = check($adminParams['7']);
            $siteAdmin['pass'] = '';
            $siteAdmin['pass_md5'] = '';
            $siteAdmin['user_id'] = 0;

            if (!empty($adminParams['3'])) {
                $siteAdmin['pass'] = check($adminParams['3']);
                $siteAdmin['pass_md5'] = md5($siteAdmin['pass']);
            }

            if (!empty($serviceParams['1'])) {
                $services = unserialize($serviceParams['1']);
                $siteAdmin['user_id'] = clean($serviceParams['3'], 'int');
            } else {
                $services[] = [
                    'service' => 0,
                    'rights_und' => check($adminParams['5']),
                    'service_time' => 0,
                    'bought_date' => '0000-00-00 00:00:00',
                    'ending_date' => '0000-00-00 00:00:00',
                    'irretrievable' => 0,
                ];
            }

            $this->insertAdmin($server->id, $siteAdmin, $services);
        }

        fclose($file);
        unlink($tempFile);

        \FtpConnection::close($connection);

        return [true, null];
    }

    public function exportAdmins($connection, Server $server, array $admins): array
    {
        return $this->export($connection, $server, $admins);
    }

    public function isImmunityUsing(): bool
    {
        return false;
    }

    public function isGroupsUsing(): bool
    {
        return false;
    }

    public function isNickWithPasswordAuthAllowed(): bool
    {
        return true;
    }

    public function isSteamIdAuthAllowed(): bool
    {
        return true;
    }

    public function isSteamIdWithPasswordAuthAllowed(): bool
    {
        return true;
    }

    /**
     * @param Admin[] $admins
     * @param mixed   $connection
     *
     * @return array{bool, ?Error}
     */
    private function export($connection, Server $server, array $admins): array
    {
        $content = $this->generateFileContent($admins);

        if (!\FtpConnection::isFileExists($connection, $server->ftp_string)) {
            return [false, error('Не найден файл на FTP сервера')];
        }

        $tempFile = __DIR__.'/../../../../../files/temp/users'.rand().'.txt';

        if (!$tempFileStream = fopen($tempFile, 'w')) {
            return [false, error('Не удалось создать временный файл')];
        }

        if (!ftp_fget($connection, $tempFileStream, $server->ftp_string, FTP_ASCII)) {
            return [false, error('Не удалось получить файл с FTP сервера')];
        }

        fclose($tempFileStream);

        file_put_contents($tempFile, $content, LOCK_EX);
        $tempFileStream = fopen($tempFile, 'r');

        if (!ftp_fput($connection, $server->ftp_string, $tempFileStream, FTP_ASCII)) {
            return [false, error('Не удалось сохранить файл на FTP сервере')];
        }

        fclose($tempFileStream);
        unlink($tempFile);

        \FtpConnection::close($connection);

        return [true, null];
    }

    /**
     * @param Admin[] $admins
     */
    private function generateFileContent(array $admins): string
    {
        $content = '';

        foreach ($admins as $admin) {
            $admin_services = $this->getAdminServices($admin->id);
            $rights = $this->getAdminRights($admin_services);
            $rights['flags'] = $this->collectAdminRights($rights['flags']);
            $admin_services = $this->serializeAdminInfo($admin_services);

            $name = htmlspecialchars_decode($admin->name, ENT_QUOTES);

            $password = '';
            if (!empty($admin->pass)) {
                $password = htmlspecialchars_decode($admin->pass, ENT_QUOTES);
            }

            $flags = $rights['flags'];
            $type = $admin->type;
            $userId = $admin->user_id;

            $content .= "\"{$name}\" \"{$password}\" \"{$flags}\" \"{$type}\"; '{$admin_services}' '{$userId}' written by gamecms :end: \n";
        }

        return $content;
    }
}
