<?php

namespace GameCMS\Integrations\Admins;

use GameCMS\Models\Admin;
use GameCMS\Models\Server;

class SourceBansIntegration extends DatabasedIntegration implements AdminSystemIntegration
{
    use AdminSystemHelpers;

    public function isBlank(): bool
    {
        return false;
    }

    public function removeAdmin(string $adminFindName, $connection, Server $server): array
    {
        $adminId = $this->getAdminIdFromStorage(
            $connection,
            $adminFindName,
            $server->db_prefix,
            $server->ip,
            $server->port
        );

        if (!$adminId) {
            return [false, error('Админ не найден в базе сервера')];
        }

        $table = set_prefix($server->db_prefix, 'admins');
        $STH = $connection->prepare("DELETE FROM `{$table}` WHERE `aid`=:id LIMIT 1");
        $STH->execute([':id' => $adminId]);

        $table = set_prefix($server->db_prefix, 'admins_servers_groups');
        $STH = $connection->prepare("DELETE FROM `{$table}` WHERE `admin_id`=:id LIMIT 1");
        $STH->execute([':id' => $adminId]);

        return [true, null];
    }

    public function exportAdmin(string $adminFindName, $connection, Server $server, Admin $admin): array
    {
        $adminServices = $this->getAdminServices($admin->id);
        $rights = $this->getAdminRights($adminServices);
        $rights['flags'] = $this->collectAdminRights($rights['flags']);
        $rights['expired'] = $this->collectAdminExpireDate($adminServices);
        $rights['immunity'] = $this->collectAdminImmunity($adminServices);
        $adminServices = $this->serializeAdminInfo($adminServices);

        $adminId = $this->getAdminIdFromStorage(
            $connection,
            $adminFindName,
            $server->db_prefix,
            $server->ip,
            $server->port
        );

        if (empty($rights['sb_group'])) {
            $rights['sb_group'] = null;
        }

        if (empty($rights['flags'])) {
            $rights['flags'] = null;
        }

        if (empty($admin->pass) || 'ce' == $admin->type) {
            $admin->pass_md5 = md5(rand());
            $admin->pass = null;
        }

        if (empty($admin->email)) {
            $admin->email = '';
        }

        if (empty($admin->nick)) {
            $admin->nick = 'unnamed';
        }

        $table = set_prefix($server->db_prefix, 'admins');
        $isExpiredColumnExists = $this->isExpiredColumnExists($connection, $table);
        if ($adminId) {
            if ($isExpiredColumnExists) {
                $connection
                    ->prepare(
                        "UPDATE `{$table}` SET `nick`=:nick, `expired`=:expired, `authid`=:authid, `immunity`=:immunity, `srv_group`=:srv_group, `srv_flags`=:srv_flags, `srv_password`=:srv_password, `gamecms`=:gamecms, `user_id`=:user_id WHERE `aid`=:id LIMIT 1"
                    )
                    ->execute(
                        [
                            ':nick' => $admin->nick,
                            ':expired' => $rights['expired'],
                            ':authid' => $admin->name,
                            ':immunity' => $rights['immunity'],
                            ':srv_group' => $rights['sb_group'],
                            ':srv_flags' => $rights['flags'],
                            ':srv_password' => $admin->pass,
                            ':gamecms' => $adminServices,
                            ':user_id' => $admin->user_id,
                            ':id' => $adminId,
                        ]
                    );
            } else {
                $connection
                    ->prepare(
                        "UPDATE `{$table}` SET `nick`=:nick, `authid`=:authid, `immunity`=:immunity, `srv_group`=:srv_group, `srv_flags`=:srv_flags, `srv_password`=:srv_password, `gamecms`=:gamecms, `user_id`=:user_id WHERE `aid`=:id LIMIT 1"
                    )
                    ->execute(
                        [
                            ':nick' => $admin->nick,
                            ':authid' => $admin->name,
                            ':immunity' => $rights['immunity'],
                            ':srv_group' => $rights['sb_group'],
                            ':srv_flags' => $rights['flags'],
                            ':srv_password' => $admin->pass,
                            ':gamecms' => $adminServices,
                            ':user_id' => $admin->user_id,
                            ':id' => $adminId,
                        ]
                    );
            }
        } else {
            $admin->user = $this->generateUniqueUserNick($admin->nick, $connection, $table);

            if ($isExpiredColumnExists) {
                $connection
                    ->prepare(
                        "INSERT INTO `{$table}` (`user`,`expired`,`nick`,`authid`,`password`,`gid`,`email`,`extraflags`,`immunity`,`srv_group`,`srv_flags`,`srv_password`,`gamecms`,`user_id`) values (:user, :expired, :nick, :authid, :password, :gid, :email, :extraflags, :immunity, :srv_group, :srv_flags, :srv_password, :gamecms, :user_id)"
                    )
                    ->execute(
                        [
                            ':user' => $admin->user,
                            ':expired' => $rights['expired'],
                            ':nick' => $admin->nick,
                            ':authid' => $admin->name,
                            ':password' => $admin->pass_md5,
                            ':gid' => '-1',
                            ':email' => $admin->email,
                            ':extraflags' => '0',
                            ':immunity' => $rights['immunity'],
                            ':srv_group' => $rights['sb_group'],
                            ':srv_flags' => $rights['flags'],
                            ':srv_password' => $admin->pass,
                            ':gamecms' => $adminServices,
                            ':user_id' => $admin->user_id,
                        ]
                    );
            } else {
                $connection
                    ->prepare(
                        "INSERT INTO `{$table}` (`user`,`nick`,`authid`,`password`,`gid`,`email`,`extraflags`,`immunity`,`srv_group`,`srv_flags`,`srv_password`,`gamecms`,`user_id`) values (:user, :nick, :authid, :password, :gid, :email, :extraflags, :immunity, :srv_group, :srv_flags, :srv_password, :gamecms, :user_id)"
                    )
                    ->execute(
                        [
                            ':user' => $admin->user,
                            ':nick' => $admin->nick,
                            ':authid' => $admin->name,
                            ':password' => $admin->pass_md5,
                            ':gid' => '-1',
                            ':email' => $admin->email,
                            ':extraflags' => '0',
                            ':immunity' => $rights['immunity'],
                            ':srv_group' => $rights['sb_group'],
                            ':srv_flags' => $rights['flags'],
                            ':srv_password' => $admin->pass,
                            ':gamecms' => $adminServices,
                            ':user_id' => $admin->user_id,
                        ]
                    );
            }

            $table = set_prefix($server->db_prefix, 'servers');
            $STH = $connection->prepare(
                "SELECT `sid` FROM `{$table}` WHERE `ip`=:ip AND `port`=:port LIMIT 1"
            );
            $STH->execute([':ip' => $server->ip, ':port' => $server->port]);
            $servers = $STH->fetchObject();

            $adminId = get_ai($connection, set_prefix($server->db_prefix, 'admins'), 'aid') - 1;
            $table = set_prefix($server->db_prefix, 'admins_servers_groups');

            $connection
                ->prepare(
                    "INSERT INTO `{$table}` (`admin_id`,`server_id`,`group_id`,`srv_group_id`) values (:admin_id, :server_id, :group_id, :srv_group_id)"
                )
                ->execute(
                    [
                        ':admin_id' => $adminId,
                        ':server_id' => $servers->sid,
                        'group_id' => '0',
                        'srv_group_id' => '-1',
                    ]
                );
        }

        return [true, null];
    }

    public function importAdmins($connection, Server $server): array
    {
        $table = set_prefix($server->db_prefix, 'servers');

        $STH = $connection->prepare("SELECT sid FROM {$table} WHERE ip=:ip AND port=:port LIMIT 1");
        $STH->execute([':ip' => $server->ip, ':port' => $server->port]);
        $row = $STH->fetchObject();
        if (empty($row->sid)) {
            return [false, error('Сервер не найден')];
        }

        $adminsServersTable = set_prefix($server->db_prefix, 'admins_servers_groups');
        $adminsTable = set_prefix($server->db_prefix, 'admins');
        $STH = $connection->query(
            "SELECT 
					{$adminsServersTable}.admin_id,
					{$adminsTable}.* 
				FROM {$adminsServersTable} 
					LEFT JOIN {$adminsTable} ON {$adminsTable}.aid = {$adminsServersTable}.admin_id 
				WHERE {$adminsServersTable}.server_id = {$row->sid}"
        );
        $STH->execute();
        $admins = $STH->fetchAll(\PDO::FETCH_OBJ);
        foreach ($admins as $serverAdmin) {
            if (empty($serverAdmin->authid)) {
                continue;
            }

            $siteAdmin = [];
            $services = [];

            $siteAdmin['name'] = check($serverAdmin->authid);
            $siteAdmin['pass'] = '';
            $siteAdmin['pass_md5'] = '';
            $siteAdmin['type'] = 'ce';
            $siteAdmin['user_id'] = 0;

            if (!empty($serverAdmin->srv_password)) {
                $siteAdmin['pass'] = $serverAdmin->srv_password;
                $siteAdmin['pass_md5'] = md5($serverAdmin->srv_password);
                $siteAdmin['type'] = 'ca';
            }

            if (!empty($serverAdmin->user_id)) {
                $siteAdmin['user_id'] = check($serverAdmin->user_id, 'int');
            }

            if (!empty($serverAdmin->gamecms)) {
                $services = unserialize($serverAdmin->gamecms);
            } else {
                $immunity = check($serverAdmin->immunity);
                $group = check($serverAdmin->srv_group);
                $flags = check($serverAdmin->srv_flags);
                $boughtDate = '0000-00-00 00:00:00';
                $endingDate = '0000-00-00 00:00:00';

                if ($flags) {
                    $services[] = [
                        'service' => 0,
                        'service_time' => 0,
                        'irretrievable' => 0,
                        'bought_date' => $boughtDate,
                        'ending_date' => $endingDate,
                        'rights_und' => $flags,
                        'immunity_und' => $immunity,
                        'sb_group_und' => '',
                    ];
                }

                if ($group) {
                    $services[] = [
                        'service' => 0,
                        'service_time' => 0,
                        'irretrievable' => 0,
                        'bought_date' => $boughtDate,
                        'ending_date' => $endingDate,
                        'rights_und' => '',
                        'immunity_und' => $immunity,
                        'sb_group_und' => $group,
                    ];
                }
            }

            $this->insertAdmin($server->id, $siteAdmin, $services);
        }

        return [true, null];
    }

    public function exportAdmins($connection, Server $server, array $admins): array
    {
        foreach ($admins as $admin) {
            [$isSuccess, $error] = $this->exportAdmin($admin->name, $connection, $server, $admin);
            if (!$isSuccess) {
                return [false, $error];
            }
        }

        return [true, null];
    }

    public function isImmunityUsing(): bool
    {
        return true;
    }

    public function isGroupsUsing(): bool
    {
        return true;
    }

    public function isNickWithPasswordAuthAllowed(): bool
    {
        return false;
    }

    public function isSteamIdAuthAllowed(): bool
    {
        return true;
    }

    public function isSteamIdWithPasswordAuthAllowed(): bool
    {
        return true;
    }

    private function getAdminIdFromStorage(
        \PDO $connection,
        string $adminFindName,
        string $dbPrefix,
        string $ip,
        int $port
    ): ?int {
        $table = set_prefix($dbPrefix, 'servers');
        $STH = $connection->prepare("SELECT `sid` FROM `{$table}` WHERE `ip`=:ip AND `port`=:port LIMIT 1");
        $STH->execute([':ip' => $ip, ':port' => $port]);
        $servers = $STH->fetchObject();
        if (empty($servers)) {
            return 0;
        }

        $table = set_prefix($dbPrefix, 'admins');
        $STH = $connection->prepare("SELECT `aid` FROM `{$table}` WHERE `authid`=:steamid");
        $STH->execute([':steamid' => $adminFindName]);
        $admins = $STH->fetchAll();
        $count = count($admins);

        $table = set_prefix($dbPrefix, 'admins_servers_groups');
        for ($i = 0; $i < $count; ++$i) {
            $STH = $connection->prepare(
                "SELECT `admin_id` FROM `{$table}` WHERE `admin_id`=:admin_id AND `server_id`=:server_id LIMIT 1"
            );
            $STH->execute([':admin_id' => $admins[$i]['aid'], ':server_id' => $servers->sid]);
            $row = $STH->fetchObject();
            if (isset($row->admin_id)) {
                return $admins[$i]['aid'];
            }
        }

        return 0;
    }

    private function isExpiredColumnExists(\PDO $pdo, string $table): bool
    {
        return check_column($table, $pdo, 'expired');
    }

    private function generateUniqueUserNick(string $nick, \PDO $connection, string $table): string
    {
        $i = 0;
        $user = $nick;

        do {
            if (0 != $i) {
                $user = $nick.'('.$i.')';
            }
            $STH = $connection->prepare("SELECT `aid` FROM `{$table}` WHERE `user`=:user LIMIT 1");
            $STH->execute([':user' => $user]);
            $row = $STH->fetchObject();
            if (isset($row->aid)) {
                $temp = null;
            } else {
                $temp = 1;
            }
            ++$i;
        } while (empty($temp));

        return $user;
    }
}
