<?php

namespace GameCMS\Integrations\Admins;

use GameCMS\Common\Error;
use GameCMS\Models\Admin;
use GameCMS\Models\Server;

class IksIntegration extends DatabasedIntegration implements AdminSystemIntegration
{
    use AdminSystemHelpers;

    public function isBlank(): bool
    {
        return false;
    }

    public function removeAdmin(string $adminFindName, $connection, Server $server): array
    {
        [$adminId, $error] = $this->getAdminIdFromStorage(
            $connection,
            $adminFindName,
            $server->db_prefix,
            $server->integration_server_id,
        );

        if ($error) {
            return [false, $error];
        }

        if (!$adminId) {
            return [false, error('Админ не найден в базе сервера', 'iks_remove_admin_not_found')];
        }

        $table = set_prefix($server->db_prefix, 'admins');
        $STH = $connection->prepare("DELETE FROM `{$table}` WHERE `id`=:id LIMIT 1");
        $STH->execute([':id' => $adminId]);

        return [true, null];
    }

    public function exportAdmin(string $adminFindName, $connection, Server $server, Admin $admin): array
    {
        $adminServices = $this->getAdminServices($admin->id);
        $rights = $this->getAdminRights($adminServices);
        $rights['flags'] = $this->collectAdminRights($rights['flags']);
        $rights['expired'] = $this->collectAdminExpireDate($adminServices);
        $rights['immunity'] = $this->collectAdminImmunity($adminServices);
        $adminServices = $this->serializeAdminInfo($adminServices);

        [$adminId, $error] = $this->getAdminIdFromStorage(
            $connection,
            $adminFindName,
            $server->db_prefix,
            $server->integration_server_id,
        );

        if ($error) {
            return [false, $error];
        }

        if (empty($rights['sb_group'])) {
            $rights['sb_group'] = '-1';
        } else {
            $table = set_prefix($server->db_prefix, 'groups');
            $STH = $connection->prepare("SELECT `id` FROM `{$table}` WHERE `name`=:name LIMIT 1");
            $STH->execute([':name' => $rights['sb_group']]);
            $group = $STH->fetchObject();

            if (!$group) {
                return [false, error('Не найдена группа в базе сервера')];
            }

            $rights['sb_group'] = $group->id;
        }

        if (empty($rights['flags'])) {
            $rights['flags'] = '';
        }

        if (empty($rights['immunity'])) {
            $rights['immunity'] = '-1';
        }

        if (empty($admin->nick)) {
            $admin->nick = 'unnamed';
        }

        $admin->name = (new \SteamIDOperations())->GetSteamID64($admin->name);

        $table = set_prefix($server->db_prefix, 'admins');

        if ($adminId) {
            $connection
                ->prepare(
                    "UPDATE `{$table}` SET `sid`=:sid, `name`=:name, `flags`=:flags, `immunity`=:immunity, `group_id`=:group_id, `end`=:end, `server_id`=:server_id, `gamecms`=:gamecms, `user_id`=:user_id WHERE `id`=:id LIMIT 1"
                )
                ->execute(
                    [
                        ':sid' => $admin->name,
                        ':name' => $admin->nick,
                        ':flags' => $rights['flags'],
                        ':immunity' => $rights['immunity'],
                        ':group_id' => $rights['sb_group'],
                        ':end' => $rights['expired'],
                        ':server_id' => $server->integration_server_id,
                        ':gamecms' => $adminServices,
                        ':user_id' => $admin->user_id,
                        ':id' => $adminId,
                    ]
                );
        } else {
            $admin->nick = $this->generateUniqueUserNick($admin->nick, $connection, $table);

            $connection
                ->prepare(
                    "INSERT INTO `{$table}` (`sid`,`name`,`flags`,`immunity`,`group_id`,`end`,`server_id`,`gamecms`,`user_id`) values (:sid, :name, :flags, :immunity, :group_id, :end, :server_id, :gamecms, :user_id)"
                )
                ->execute(
                    [
                        ':sid' => $admin->name,
                        ':name' => $admin->nick,
                        ':flags' => $rights['flags'],
                        ':immunity' => $rights['immunity'],
                        ':group_id' => $rights['sb_group'],
                        ':end' => $rights['expired'],
                        ':server_id' => $server->integration_server_id,
                        ':gamecms' => $adminServices,
                        ':user_id' => $admin->user_id,
                    ]
                );
        }

        return [true, null];
    }

    public function importAdmins($connection, Server $server): array
    {
        $table = set_prefix($server->db_prefix, 'admins');

        $STH = $connection->query("SELECT * FROM {$table} WHERE server_id LIKE '%;%'");
        $groupedAdmins = $STH->fetchAll(\PDO::FETCH_OBJ);
        foreach ($groupedAdmins as $groupedAdmin) {
            $connection->exec("DELETE FROM {$table} WHERE id = {$groupedAdmin->id} LIMIT 1");

            $serverIds = explode(';', $groupedAdmin->server_id);

            foreach ($serverIds as $serverId) {
                $connection->prepare(
                    "INSERT INTO {$table} (sid, name, flags, immunity, group_id, end, server_id, gamecms, user_id) values (:sid, :name, :flags, :immunity, :group_id, :end, :server_id, :gamecms, :user_id)"
                )
                    ->execute(
                        [
                            'sid' => $groupedAdmin->sid,
                            'name' => $groupedAdmin->name,
                            'flags' => $groupedAdmin->flags,
                            'immunity' => $groupedAdmin->immunity,
                            'group_id' => $groupedAdmin->group_id,
                            'end' => $groupedAdmin->end,
                            'server_id' => $serverId,
                            'gamecms' => $groupedAdmin->gamecms,
                            'user_id' => $groupedAdmin->user_id,
                        ]
                    );
            }
        }

        $STH = $connection->prepare("SELECT * FROM {$table} WHERE server_id = :server_id");
        $STH->execute([':server_id' => $server->integration_server_id]);
        $admins = $STH->fetchAll(\PDO::FETCH_OBJ);
        foreach ($admins as $serverAdmin) {
            if (empty($serverAdmin->sid)) {
                continue;
            }

            $siteAdmin = [];
            $services = [];

            $siteAdmin['name'] = (new \SteamIDOperations())->GetSteamID32($serverAdmin->sid);
            $siteAdmin['type'] = 'ce';
            $siteAdmin['user_id'] = 0;

            if (!empty($serverAdmin->user_id)) {
                $siteAdmin['user_id'] = check($serverAdmin->user_id, 'int');
            }

            if (!empty($serverAdmin->gamecms)) {
                $services = unserialize($serverAdmin->gamecms);
            } else {
                $group = 0;
                if ('-1' != $serverAdmin->group_id) {
                    $groupsTable = set_prefix($server->db_prefix, 'groups');

                    $STH = $connection->prepare("SELECT name FROM {$groupsTable} WHERE id = :group_id LIMIT 1");
                    $STH->execute([':group_id' => $serverAdmin->group_id]);
                    $group = $STH->fetchObject();

                    $group = check($group->name, null);
                }

                $immunity = 0;
                if ('-1' != $serverAdmin->immunity) {
                    $immunity = check($serverAdmin->immunity, 'int');
                }

                $boughtDate = '0000-00-00 00:00:00';
                $endingDate = '0000-00-00 00:00:00';

                if (0 != $serverAdmin->end) {
                    $endingDate = date('Y-m-d H:i:s', $serverAdmin->end);
                }

                $flags = check($serverAdmin->flags);

                if ($flags) {
                    $services[] = [
                        'service' => 0,
                        'service_time' => 0,
                        'irretrievable' => 0,
                        'bought_date' => $boughtDate,
                        'ending_date' => $endingDate,
                        'rights_und' => $flags,
                        'immunity_und' => $immunity,
                        'sb_group_und' => '',
                    ];
                }

                if ($group) {
                    $services[] = [
                        'service' => 0,
                        'service_time' => 0,
                        'irretrievable' => 0,
                        'bought_date' => $boughtDate,
                        'ending_date' => $endingDate,
                        'rights_und' => '',
                        'immunity_und' => $immunity,
                        'sb_group_und' => $group,
                    ];
                }
            }

            $this->insertAdmin($server->id, $siteAdmin, $services);
        }

        return [true, null];
    }

    public function exportAdmins($connection, Server $server, array $admins): array
    {
        foreach ($admins as $admin) {
            [$isSuccess, $error] = $this->exportAdmin($admin->name, $connection, $server, $admin);
            if (!$isSuccess) {
                return [false, $error];
            }
        }

        return [true, null];
    }

    public function isImmunityUsing(): bool
    {
        return true;
    }

    public function isGroupsUsing(): bool
    {
        return true;
    }

    public function isNickWithPasswordAuthAllowed(): bool
    {
        return false;
    }

    public function isSteamIdAuthAllowed(): bool
    {
        return true;
    }

    public function isSteamIdWithPasswordAuthAllowed(): bool
    {
        return false;
    }

    /**
     * @return array{int, ?Error}
     */
    private function getAdminIdFromStorage(
        \PDO $connection,
        string $adminFindName,
        string $dbPrefix,
        int $serverId
    ): array {
        $adminFindName = (new \SteamIDOperations())->GetSteamID64($adminFindName);
        if (!$adminFindName) {
            return [0, error('Не удалось получить STEAM ID 64')];
        }

        $table = set_prefix($dbPrefix, 'admins');
        $STH = $connection->prepare(
            "SELECT `id` FROM `{$table}` WHERE `sid`=:steam_id AND `server_id`=:server_id LIMIT 1"
        );
        $STH->execute([':steam_id' => $adminFindName, ':server_id' => $serverId]);
        $admin = $STH->fetchObject();

        return [$admin ? $admin->id : 0, null];
    }

    private function generateUniqueUserNick(string $nick, \PDO $connection, string $table): string
    {
        $i = 0;
        $user = $nick;

        do {
            if (0 != $i) {
                $user = $nick.'('.$i.')';
            }
            $STH = $connection->prepare("SELECT `id` FROM `{$table}` WHERE `name`=:name LIMIT 1");
            $STH->execute([':name' => $user]);
            $row = $STH->fetchObject();
            if (isset($row->aid)) {
                $temp = null;
            } else {
                $temp = 1;
            }
            ++$i;
        } while (empty($temp));

        return $user;
    }
}
